implementation module StdPicture


//	Version 1.1

//	Drawing functions and other operations on Picture


import	StdBool, StdFunc, StdInt, StdList, StdMisc, StdReal, StdTuple
import	ospicture, osfont, osrgn, ostoolbox
import	commondef, StdPictureDef

//	MW: DrawFunction will be removed in vers. 1.3
::	DrawFunction
	:==	*Picture -> *Picture

//	The picture attributes:
::	PictureAttribute						// Default:
	=	PicturePenSize		Int				// 1
	|	PicturePenPos		Point			// zero
	|	PicturePenColour	Colour			// Black
	|	PicturePenFont		Font			// DefaultFont


//	Pen position attributes:
setPenPos :: !Point !*Picture -> *Picture
setPenPos pos picture
	= setpictpenpos pos picture

getPenPos :: !*Picture -> (!Point,!*Picture)
getPenPos picture
	= getpictpenpos picture

class movePenPos figure	:: !figure !*Picture -> *Picture
//	Move the pen position as much as when drawing the figure.

instance movePenPos Vector where
	movePenPos :: !Vector !*Picture -> *Picture
	movePenPos v picture
		= movepictpenpos v picture

instance movePenPos Curve where
	movePenPos :: !Curve !*Picture -> *Picture
	movePenPos curve picture
		# (curpos,picture)	= getpictpenpos picture
		  (_,_,endpos)		= getcurve_rect_begin_end curpos curve
		# picture			= setpictpenpos endpos picture
		= picture


//	PenSize attributes:
setPenSize :: !Int !*Picture -> *Picture
setPenSize w picture
	= setpictpensize w picture

getPenSize :: !*Picture -> (!Int,!*Picture)
getPenSize picture
	= getpictpensize picture

setDefaultPenSize :: !*Picture -> *Picture
setDefaultPenSize picture
	= setpictpensize 1 picture


//	Colour attributes:
setPenColour :: !Colour !*Picture -> *Picture
setPenColour c picture
	= setpictpencolour c picture

getPenColour :: !*Picture -> (!Colour,!*Picture)
getPenColour picture
	= getpictpencolour picture

setDefaultPenColour :: !*Picture -> *Picture
setDefaultPenColour picture
	= setpictpencolour Black picture


//	Font attributes:
setPenFont :: !Font !*Picture -> *Picture
setPenFont f picture
	= setpictpenfont f picture

getPenFont :: !*Picture -> (!Font,!*Picture)
getPenFont picture
	= getpictpenfont picture

setDefaultPenFont :: !*Picture -> *Picture
setDefaultPenFont picture
	= setpictpendefaultfont picture


//	Font operations:
openFont :: !FontDef !*Picture -> (!(!Bool,!Font),!*Picture)
openFont {fName,fStyles,fSize} picture
//	# (origin,pen,context,tb)	= unpackPicture picture
	# (origin,pen,context,tb)	= peekPicture picture
	# (found,font,tb)			= OSselectfont (fName,fStyles,fSize) tb
	# picture					= unpeekPicture origin pen context tb
//	# picture					= packPicture origin pen context tb
	= ((found,font),picture)

openDefaultFont :: !*Picture -> (!Font,!*Picture)
openDefaultFont picture
	= accpicttoolbox OSdefaultfont picture

openDialogFont :: !*Picture -> (!Font,!*Picture)
openDialogFont picture
	= accpicttoolbox OSdialogfont picture

getFontNames :: !*Picture -> (![FontName],!*Picture)
getFontNames picture
	= accpicttoolbox OSfontnames picture

getFontStyles :: !FontName	!*Picture -> (![FontStyle],!*Picture)
getFontStyles fName picture
	= accpicttoolbox (OSfontstyles fName) picture

getFontSizes :: !Int !Int !FontName	!*Picture -> (![FontSize],!*Picture)
getFontSizes sizeBound1 sizeBound2 fName picture
	= accpicttoolbox (OSfontsizes sizeBound1 sizeBound2 fName) picture

getFontDef :: !Font -> FontDef
getFontDef font
	= {fName=name,fStyles=styles,fSize=size}
where
	(name,styles,size)	= OSfontgetdef font

getFontCharWidth :: !Font !Char !*Picture -> (!Int,!*Picture)
getFontCharWidth font char picture
	# (osPictContext,picture)	= peekOSPictContext picture
	# (widths,picture)			= accpicttoolbox (OSgetfontcharwidths True osPictContext [char] font) picture
	= (hd widths,picture)

getFontCharWidths :: !Font ![Char] !*Picture -> (![Int],!*Picture)
getFontCharWidths font chars picture
	# (osPictContext,picture)	= peekOSPictContext picture
	= accpicttoolbox (OSgetfontcharwidths True osPictContext chars font) picture

getFontStringWidth :: !Font !String !*Picture -> (!Int,!*Picture)
getFontStringWidth font string picture
	# (osPictContext,picture)	= peekOSPictContext picture
	# (widths,picture)			= accpicttoolbox (OSgetfontstringwidths True osPictContext [string] font) picture
	= (hd widths,picture)

getFontStringWidths :: !Font ![String] !*Picture -> (![Int],!*Picture)
getFontStringWidths font strings picture
	# (osPictContext,picture)	= peekOSPictContext picture
	= accpicttoolbox (OSgetfontstringwidths True osPictContext strings font) picture

getFontMetrics :: !Font !*Picture -> (!FontMetrics,!*Picture)
getFontMetrics font picture
	# (osPictContext,picture)						= peekOSPictContext picture
	# ((ascent,descent,leading,maxwidth),picture)	= accpicttoolbox (OSgetfontmetrics True osPictContext font) picture
	= ({fAscent=ascent,fDescent=descent,fLeading=leading,fMaxWidth=maxwidth},picture)

getPenFontCharWidth :: !Char !*Picture -> (!Int,!*Picture)
getPenFontCharWidth char picture = getPenFontInfo (\font->getFontCharWidth font char) picture

getPenFontCharWidths :: ![Char] !*Picture -> (![Int],!*Picture)
getPenFontCharWidths chars picture = getPenFontInfo (\font->getFontCharWidths font chars) picture

getPenFontStringWidth :: !String !*Picture -> (!Int,!*Picture)
getPenFontStringWidth string picture = getPenFontInfo (\font->getFontStringWidth font string) picture

getPenFontStringWidths :: ![String] !*Picture -> (![Int],!*Picture)
getPenFontStringWidths strings picture = getPenFontInfo (\font->getFontStringWidths font strings) picture

getPenFontMetrics :: !*Picture -> (!FontMetrics,!*Picture)
getPenFontMetrics picture = getPenFontInfo (\font->getFontMetrics font) picture

getPenFontInfo :: !(Font -> *Picture -> (.x,*Picture)) !*Picture -> (.x,!*Picture)
getPenFontInfo fontfun picture
	# (font,picture)	= getPenFont picture
	# (x,picture)		= fontfun font picture
	# picture			= setPenFont font picture
	= (x,picture)


/*	Drawing functions.
	These functions are divided into the following classes:
	Drawables:
		draw     'line-oriented' figures at the current  pen position.
		drawAt   'line-oriented' figures at the argument pen position.
		undraw     f = appPicture (draw     f o setPenColour background)
		undrawAt x f = appPicture (drawAt x f o setPenColour background)
	Fillables:
		fill     'area-oriented' figures at the current  pen position.
		fillAt   'area-oriented' figures at the argument pen position.
		unfill     f = appPicture (fill     f o setPenColour background)
		unfillAt x f = appPicture (fillAt x f o setPenColour background)
	Hilites:
		hilite	 draws figures in the appropriate 'hilite' mode at the current pen position.
		hiliteAt draws figures in the appropriate 'hilite' mode at the current pen position.
		Both functions reset the 'hilite' after drawing.
*/
class Drawables figure where
	draw	::			!figure					!*Picture -> *Picture
	drawAt	:: !Point	!figure					!*Picture -> *Picture
	undraw	::			!figure					!*Picture -> *Picture
	undrawAt:: !Point	!figure					!*Picture -> *Picture

class Fillables figure where
	fill	::			!figure					!*Picture -> *Picture
	fillAt	:: !Point	!figure					!*Picture -> *Picture
	unfill	::			!figure					!*Picture -> *Picture
	unfillAt:: !Point	!figure					!*Picture -> *Picture

class Hilites figure where
	hilite	::			!figure					!*Picture -> *Picture
	hiliteAt:: !Point	!figure					!*Picture -> *Picture


/*	(app/acc)Picture applies the given drawing function to the given picture.
	When drawing is done, all picture attributes are set to the attribute values of the original picture.
*/
appPicture :: !(IdFun *Picture) !*Picture -> *Picture
appPicture drawf picture
	# (pen,picture)	= getpictpen picture
	# picture		= drawf picture
	# picture		= setpictpen pen picture
	= picture

accPicture :: !(St *Picture .x) !*Picture -> (.x,!*Picture)
accPicture drawf picture
	# (pen,picture)	= getpictpen picture
	# (x,picture)	= drawf picture
	# picture		= setpictpen pen picture
	= (x,picture)


//	Drawing in a clipping region.

::	Region
	=	{	region_shape	:: ![RegionShape]
		,	region_bound	:: !Rect
		}
::	RegionShape
	=	RegionRect		Rect
	|	RegionPolygon	(Int,Int) [(Int,Int)]


isEmptyRegion :: !Region -> Bool
isEmptyRegion {region_shape=[]}	= True
isEmptyRegion _					= False

getRegionBound :: !Region -> Rectangle
getRegionBound {region_bound} = RectToRectangle region_bound

class toRegion area :: !area -> Region

::	PolygonAt
	=	{	polygon_pos	:: !Point
		,	polygon		:: !Polygon
		}

instance toRegion Rectangle where
	toRegion :: !Rectangle -> Region
	toRegion rectangle
		| IsEmptyRect rect	= zero
		| otherwise			= {region_shape=[RegionRect rect],region_bound=rect}
	where
		rect				= RectangleToRect rectangle

instance toRegion PolygonAt where
	toRegion :: !PolygonAt -> Region
	toRegion {polygon_pos={x,y},polygon={polygon_shape}}
		| IsEmptyRect bound	= zero
		| otherwise			= {region_shape=[RegionPolygon (x,y) shape],region_bound=bound}
	where
		shape				= tupleShape zero polygon_shape
		bound				= polybound (x,y) shape (x,y,x,y)
		
		polybound :: !(!Int,!Int) ![(Int,Int)] !Rect -> Rect
		polybound (x,y) [(vx,vy):vs] (minx,miny,maxx,maxy)
			= polybound (x`,y`) vs (minx`,miny`,maxx`,maxy`)
		where
			(x`,y`)	= (x+vx,y+vy)
			minx`	= min minx x`;	miny`	= min miny y`;
			maxx`	= max maxx x`;	maxy`	= max maxy y`;
		polybound _ _ bound
			= bound
		
		tupleShape :: !Vector ![Vector] -> [(Int,Int)]
		tupleShape v []
			| v==zero	= []
			| otherwise	= [(0-v.vx,0-v.vy)]
		tupleShape v [v`=:{vx,vy}:vs]
			= [(vx,vy):tupleShape (v+v`) vs]

instance toRegion [area] | toRegion area where
	toRegion :: ![area] -> Region	| toRegion area
	toRegion [area:areas]	= toRegion area + toRegion areas
	toRegion _				= zero

instance toRegion (:^: area1 area2)	| toRegion area1 & toRegion area2 where
	toRegion :: !(:^: area1 area2) -> Region | toRegion area1 & toRegion area2
	toRegion (r1 :^: r2) = toRegion r1 + toRegion r2

instance zero Region where
	zero :: !Region
	zero = {region_shape=[],region_bound=ZeroRect}
instance + Region where
	(+) :: !Region !Region -> Region
	(+) r1 r2
	| IsEmptyRect r1.region_bound
		= r2
	| IsEmptyRect r2.region_bound
		= r1
	| otherwise
		= {region_shape=r1.region_shape++r2.region_shape,region_bound=sumbound r1.region_bound r2.region_bound}
	where
		sumbound :: !Rect !Rect -> Rect
		sumbound (minx,miny,maxx,maxy) (minx`,miny`,maxx`,maxy`) = (min minx minx`,min miny miny`,max maxx maxx`,max maxy maxy`)

appClipPicture :: !Region !(IdFun *Picture) !*Picture -> *Picture
appClipPicture region drawf picture
	= snd (accClipPicture region (\p->(undef,drawf p)) picture)

accClipPicture :: !Region !(St *Picture .x) !*Picture -> (.x,!*Picture)
accClipPicture region drawf picture
	# (curClipRgn,picture)		= pictgetcliprgn picture
//	# (origin,pen,context,tb)	= unpackPicture picture
	# (origin,pen,context,tb)	= peekPicture picture
	# (newClipRgn,tb)			= osnewrgn tb
	# (hFac,vFac,context,tb)	= getPictureScalingFactors context tb
	# (newClipRgn,tb)			= setrgnshapes hFac vFac origin region.region_shape newClipRgn tb
	# picture					= unpeekPicture origin pen context tb
//	# picture					= packPicture origin pen context tb
	  (set,dispose)				= if (curClipRgn==0) (pictsetcliprgn,\_ x->x) (pictandcliprgn,osdisposergn)
	# picture					= set newClipRgn picture
//	# picture					= pictandcliprgn newClipRgn picture
	# (x,picture)				= drawf picture
	# picture					= pictsetcliprgn curClipRgn picture
	# picture					= apppicttoolbox (osdisposergn newClipRgn) picture
	# picture					= apppicttoolbox (dispose curClipRgn) picture
	= (x,picture)
where
	setrgnshapes :: !(!Int,!Int) !(!Int,!Int) !Point ![RegionShape] !OSRgnHandle !*OSToolbox -> (!OSRgnHandle,!*OSToolbox)
	setrgnshapes hFac vFac origin [shape:shapes] rgn tb
		# (rgn,tb)				= setrgnshape hFac vFac origin shape rgn tb
		= setrgnshapes hFac vFac origin shapes rgn tb
	where
		setrgnshape :: !(!Int,!Int) !(!Int,!Int) !Point !RegionShape !OSRgnHandle !*OSToolbox -> (!OSRgnHandle,!*OSToolbox)
		setrgnshape hFac vFac {x=ox,y=oy} (RegionRect (left,top,right,bottom)) rgn tb
			# (emptyrgn,tb)	= osnewrgn tb
			# (rectrgn, tb)	= osrectrgn rect emptyrgn tb
			# (sumrgn,  tb)	= osunionrgn rectrgn rgn tb
			# tb			= osdisposergn rectrgn tb
			# tb			= osdisposergn rgn tb
			= (sumrgn,tb)
		where
			rect			= (scale hFac (left-ox),scale vFac (top-oy),scale hFac (right-ox),scale vFac (bottom-oy))
		setrgnshape hFac vFac {x=ox,y=oy} (RegionPolygon (x,y) shape) rgn tb
			# (emptyrgn,tb)	= osnewrgn tb
			# (polyrgn, tb)	= ospolyrgn (scale hFac (x-ox),scale vFac (y-oy)) (map (\(vx,vy)->(scale hFac vx,scale vFac vy)) shape) emptyrgn tb
			# (sumrgn,  tb)	= osunionrgn polyrgn rgn tb
			# tb			= osdisposergn polyrgn tb
			# tb			= osdisposergn rgn tb
			= (sumrgn,tb)
	setrgnshapes _ _ _ _ rgn tb
		= (rgn,tb)
	
	scale :: !(!Int,!Int) !Int -> Int
	scale (n,d) x = n*x/d


/*	(app/acc)XorPicture applies the given drawing function to the given picture in the platform appropriate
	xor mode. 
*/
appXorPicture :: !(IdFun *Picture) !*Picture -> *Picture
appXorPicture drawf picture
	# picture			= setpictxormode picture
	# picture			= drawf picture
	# picture			= setpictnormalmode picture
	= picture

accXorPicture :: !(St *Picture .x) !*Picture -> (.x,!*Picture)
accXorPicture drawf picture
	# picture			= setpictxormode picture
	# (x,picture)		= drawf picture
	# picture			= setpictnormalmode picture
	= (x,picture)


/*	Hiliting figures:
*/
instance Hilites Box where
	hilite :: !Box !*Picture -> *Picture
	hilite box picture
		# picture			= setpicthilitemode picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictfillrect (boxtorect curpos box) picture
		# picture			= setpictnormalmode picture
		= picture
	
	hiliteAt :: !Point !Box !*Picture -> *Picture
	hiliteAt base box picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (boxtorect base box) picture
		# picture	= setpictnormalmode picture
		= picture

instance Hilites Rectangle where
	hilite :: !Rectangle !*Picture -> *Picture
	hilite rectangle picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (RectangleToRect rectangle) picture
		# picture	= setpictnormalmode picture
		= picture
	
	hiliteAt :: !Point !Rectangle !*Picture -> *Picture
	hiliteAt _ rectangle picture
		# picture	= setpicthilitemode picture
		# picture	= pictfillrect (RectangleToRect rectangle) picture
		# picture	= setpictnormalmode picture
		= picture


drawPoint :: !*Picture -> *Picture
drawPoint picture
	# (curpos,picture)	= getpictpenpos picture
	# picture			= pictdrawpoint curpos picture
	# picture			= setpictpenpos {curpos & x=curpos.x+1} picture
	= picture

drawPointAt :: !Point !*Picture -> *Picture
drawPointAt point picture
	# (curpos,picture)	= getpictpenpos picture
	# picture			= pictdrawpoint point picture
	# picture			= setpictpenpos curpos picture
	= picture


/*	Point connecting drawing operations:
*/
drawLineTo :: !Point !*Picture -> *Picture
drawLineTo pos picture
	= pictdrawlineto pos picture

drawLine :: !Point !Point !*Picture -> *Picture
drawLine pos1 pos2 picture
	= pictdrawline pos1 pos2 picture


/*	Text drawing operations:
*/
instance Drawables Char where
	draw :: !Char !*Picture -> *Picture
	draw char picture
		= pictdrawchar char picture
	
	drawAt :: !Point !Char !*Picture -> *Picture
	drawAt pos char picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictdrawchar char picture
		# picture			= setpictpenpos curpos picture
		= picture
	
	undraw :: !Char !*Picture -> *Picture
	undraw char picture
		= pictundrawchar char picture
	
	undrawAt :: !Point !Char !*Picture -> *Picture
	undrawAt pos char picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictundrawchar char picture
		# picture			= setpictpenpos curpos picture
		= picture

instance Drawables {#Char} where
	draw :: !{#Char} !*Picture -> *Picture
	draw string picture
		= pictdrawstring string picture
	
	drawAt :: !Point !{#Char} !*Picture -> *Picture
	drawAt pos string picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictdrawstring string picture
		# picture			= setpictpenpos curpos picture
		= picture
	
	undraw :: !{#Char} !*Picture -> *Picture
	undraw string picture
		= pictundrawstring string picture
	
	undrawAt :: !Point !{#Char} !*Picture -> *Picture
	undrawAt pos string picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= setpictpenpos pos picture
		# picture			= pictundrawstring string picture
		# picture			= setpictpenpos curpos picture
		= picture


/*	Vector drawing operations:
*/
instance Drawables Vector where
	draw :: !Vector !*Picture -> *Picture
	draw {vx,vy} picture
		# (curpos,picture)	= getpictpenpos picture
		  endpos			= {x=curpos.x+vx,y=curpos.y+vy}
		# picture			= pictdrawlineto endpos picture
		= picture
	
	drawAt :: !Point !Vector !*Picture -> *Picture
	drawAt pos=:{x,y} {vx,vy} picture
		= pictdrawline pos {x=x+vx,y=y+vy} picture
	
	undraw :: !Vector !*Picture -> *Picture
	undraw {vx,vy} picture
		# (curpos,picture)	= getpictpenpos picture
		  endpos			= {x=curpos.x+vx,y=curpos.y+vy}
		# picture			= pictundrawlineto endpos picture
		= picture
	
	undrawAt :: !Point !Vector !*Picture -> *Picture
	undrawAt pos=:{x,y} {vx,vy} picture
		= pictundrawline pos {x=x+vx,y=y+vy} picture


/*	Oval drawing operations:
*/
instance Drawables Oval where
	draw :: !Oval !*Picture -> *Picture
	draw oval picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictdrawoval curpos oval picture
		= picture
	
	drawAt :: !Point !Oval !*Picture -> *Picture
	drawAt pos oval picture
		= pictdrawoval pos oval picture
	
	undraw :: !Oval !*Picture -> *Picture
	undraw oval picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictundrawoval curpos oval picture
		= picture
	
	undrawAt :: !Point !Oval !*Picture -> *Picture
	undrawAt pos oval picture
		= pictundrawoval pos oval picture

instance Fillables Oval where
	fill :: !Oval !*Picture -> *Picture
	fill oval picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictfilloval curpos oval picture
		= picture
	
	fillAt :: !Point !Oval !*Picture -> *Picture
	fillAt pos oval picture
		= pictfilloval pos oval picture
	
	unfill :: !Oval !*Picture -> *Picture
	unfill oval picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictunfilloval curpos oval picture
		= picture
	
	unfillAt :: !Point !Oval !*Picture -> *Picture
	unfillAt pos oval picture
		= pictunfilloval pos oval picture


/*	Curve drawing operations:
*/
instance Drawables Curve where
	draw :: !Curve !*Picture -> *Picture
	draw curve picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictdrawcurve True curpos curve picture
		= picture
	
	drawAt :: !Point !Curve !*Picture -> *Picture
	drawAt point curve picture
		= pictdrawcurve False point curve picture
	
	undraw :: !Curve !*Picture -> *Picture
	undraw curve picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictundrawcurve True curpos curve picture
		= picture
	
	undrawAt :: !Point !Curve !*Picture -> *Picture
	undrawAt point curve picture
		= pictundrawcurve False point curve picture

instance Fillables Curve where
	fill :: !Curve !*Picture -> *Picture
	fill curve picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictfillcurve True curpos curve picture
		= picture
	
	fillAt :: !Point !Curve !*Picture -> *Picture
	fillAt point curve picture
		= pictfillcurve False point curve picture

	unfill :: !Curve !*Picture -> *Picture
	unfill curve picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictunfillcurve True curpos curve picture
		= picture
	
	unfillAt :: !Point !Curve !*Picture -> *Picture
	unfillAt point curve picture
		= pictunfillcurve False point curve picture


/*	Box drawing operations:
*/
instance Drawables Box where
	draw :: !Box !*Picture -> *Picture
	draw box picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictdrawrect (boxtorect curpos box) picture
		= picture
	
	drawAt :: !Point !Box !*Picture -> *Picture
	drawAt point box picture
		= pictdrawrect (boxtorect point box) picture
	
	undraw :: !Box !*Picture -> *Picture
	undraw box picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictundrawrect (boxtorect curpos box) picture
		= picture
	
	undrawAt :: !Point !Box !*Picture -> *Picture
	undrawAt point box picture
		= pictundrawrect (boxtorect point box) picture

instance Fillables Box where
	fill :: !Box !*Picture -> *Picture
	fill box picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictfillrect (boxtorect curpos box) picture
		= picture
	
	fillAt :: !Point !Box !*Picture -> *Picture
	fillAt pos box picture
		= pictfillrect (boxtorect pos box) picture

	unfill :: !Box !*Picture -> *Picture
	unfill box picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictunfillrect (boxtorect curpos box) picture
		= picture
	
	unfillAt :: !Point !Box !*Picture -> *Picture
	unfillAt pos box picture
		= pictunfillrect (boxtorect pos box) picture

boxtorect :: !Point !Box -> (!Int,!Int,!Int,!Int)
boxtorect {x,y} {box_w,box_h}
	= (l,t, r,b)
where
	(l,r) = minmax x (x+box_w)
	(t,b) = minmax y (y+box_h)


/*	Rectangle drawing operations:
*/
instance Drawables Rectangle where
	draw :: !Rectangle !*Picture -> *Picture
	draw rectangle picture
		= pictdrawrect (RectangleToRect rectangle) picture
	
	drawAt :: !Point !Rectangle !*Picture -> *Picture
	drawAt _ rectangle picture
		= pictdrawrect (RectangleToRect rectangle) picture

	undraw :: !Rectangle !*Picture -> *Picture
	undraw rectangle picture
		= pictundrawrect (RectangleToRect rectangle) picture
	
	undrawAt :: !Point !Rectangle !*Picture -> *Picture
	undrawAt _ rectangle picture
		= pictundrawrect (RectangleToRect rectangle) picture

instance Fillables Rectangle where
	fill :: !Rectangle !*Picture -> *Picture
	fill rectangle picture
		= pictfillrect (RectangleToRect rectangle) picture
	
	fillAt :: !Point !Rectangle !*Picture -> *Picture
	fillAt _ rectangle picture
		= pictfillrect (RectangleToRect rectangle) picture

	unfill :: !Rectangle !*Picture -> *Picture
	unfill rectangle picture
		= pictunfillrect (RectangleToRect rectangle) picture
	
	unfillAt :: !Point !Rectangle !*Picture -> *Picture
	unfillAt _ rectangle picture
		= pictunfillrect (RectangleToRect rectangle) picture


/*	Polygon drawing operations:
*/
instance Drawables Polygon where
	draw :: !Polygon !*Picture -> *Picture
	draw polygon picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictdrawpolygon curpos polygon picture
		= picture
	
	drawAt :: !Point !Polygon !*Picture -> *Picture
	drawAt base polygon picture
		= pictdrawpolygon base polygon picture

	undraw :: !Polygon !*Picture -> *Picture
	undraw polygon picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictundrawpolygon curpos polygon picture
		= picture
	
	undrawAt :: !Point !Polygon !*Picture -> *Picture
	undrawAt base polygon picture
		= pictundrawpolygon base polygon picture

instance Fillables Polygon where
	fill :: !Polygon !*Picture -> *Picture
	fill polygon picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictfillpolygon curpos polygon picture
		= picture
	
	fillAt :: !Point !Polygon !*Picture -> *Picture
	fillAt base polygon picture
		= pictfillpolygon base polygon picture
	
	unfill :: !Polygon !*Picture -> *Picture
	unfill polygon picture
		# (curpos,picture)	= getpictpenpos picture
		# picture			= pictunfillpolygon curpos polygon picture
		= picture
	
	unfillAt :: !Point !Polygon !*Picture -> *Picture
	unfillAt base polygon picture
		= pictunfillpolygon base polygon picture


// MW...
getResolution :: !*Picture -> (!(!Int,!Int),!*Picture)
getResolution picture
//	# (origin,pen,context,tb)	= unpackPicture picture
	# (origin,pen,context,tb)	= peekPicture picture
	# (res,tb)					= getResolutionC context tb
	= (res,unpeekPicture origin pen context tb)
//	= (res,packPicture origin pen context tb)
// ... MW
